/** @file   filelist.h
 * @brief   Declaration of FileList - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_FILELIST_H
#define H_ENG2D_FILELIST_H

#include <string>
#include <vector>
#include <iostream>
#include "eng2d_dll.h"
#include "eng2d_Exception.h"

namespace eng2d {


/** @class  FileList
 * @brief   Loads a list of filenames from a file and provides access to them.
 * @author  Tomi Lamminsaari
 *
 * An instance of this class is a kind of vector with filenames as its
 * elements. The contents can be loaded from file or saved to a file.
 *
 * @code
 * #
 * # Here is an example file that can be loaded in FileList object
 * #
 * <eng2d_filelist>
 *     file: testfile1.txt
 *     file: testfile2.txt
 *     # Sound files
 *     file: enginesound.wav
 * </eng2d_filelist>
 * @endcode
 */
class DLLIMPORT FileList
{
public:
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Default contructor. Creates an empty filelist.
   */
  FileList();
  
	/** Constructor.
   * @param     filename          Name of the file that contains the filelist.
   */
	FileList( const std::string& filename ) throw (xIoError);


	/** Destructor
   */
	virtual ~FileList();

	/** Copy constructor.
   * @param     rO                Reference to another FileList
   */
  FileList( const FileList& rO );

	/** Assignment operator
   * @param     rO                Reference to another FileList
   * @return    Reference to us.
   */
  FileList& operator = ( const FileList& rO );


  ///
  /// Methods
  /// =======
  
  /** Adds new file to this filelist.
   * @param     filename          Filename being added.
   */
  void addFile( const std::string& filename );

  /** Clears this filelist.
   */
  void clear();
  
  /** Deletes the index'th filename
   * @param     index             Index of the filename being deleted.
   */
  void delFile( int index );
  
  /** Sets the index'th filename
   * @param     index             Index of the filename we want to replace.
   * @param     filename          New filename.
   */
  void setFile( int index, const std::string& filename );
  
  /** Deletes the first occurance of given filename.
   * @param     fname             Name of the file being removed from the list.
   * @return    <code>false</code> if we couldn't find any occurance of
   *            given filename.
   */
  void delFile( const std::string& fname );

  /** Loads the filelist from given file
   * @param     filename        Name of the list file
   * @return    One of the following: <code>KErrNone, KErrNotFound,
   *            KErrNotSupported, KErrEof, KErrCorrupt</code>
   */
  int load( const std::string& filename );
  
  /** Saves the filelist to given file.
   * @param     filename        Name of the file where we save this filelist
   * @return    Either @c KErrNone or @c KErrNotFound
   */
  int save( const std::string& filename ) const;


  ///
  /// Getter methods
  /// ==============
  
  /** Tells if we succeeded to load the filelist.
   * @return    <code>true</code> if we failed to load the filelist.
   */
  bool bad() const;
  
  /** Returns name of the index'th file from the filelist.
   * @param     index             Number of the filename being returned.
   * @return    The index'th filename or empty string if there is no
   *            filename in such index.
   */
  std::string getFile( int index ) const;

  /** Returns the number of file the filelist has.
   * @return    Number of files.
   */
  int filecount() const;
  
  /** Searches for given filename from this filelist and returns the index
   * of it.
   * @param     searchFor         Filename we search for.
   * @return    From wich index the filename can be found or KErrNotFound if
   *            there is no such entry.
   */
  int contains( const std::string& searchFor ) const;

  

protected:

  /** Tells if the given index is valid.
   * @param     index             The index
   * @return    <code>true</code> if index is within the
   *            range [0 ... numOfFiles]
   */
  bool validIndex( int index ) const;
  
  /** Reads until the opening tag or EOF is encountered.
   * @param     rIn               Input stream
   * @return    <code>true</code> if EOF was encountered before the opening tag.
   */
  bool readHeader( std::istream& rIn ) const;

  ///
  /// Members
  /// =======
  
  /** This flag is set if loading fails. */
  bool  m_bad;
  /** This vector holds the contents of the filelist. */
  std::vector< std::string >  m_filelist;



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
